import { Control } from "../../api/gui/Control.js"
import { on } from "../../lib/event/on.js"
import { positionable } from "../../api/gui/trait/positionable.js"
import "../layout/listbox.js"

/** @import { ListboxComponent } from "../layout/listbox.js" */

/**
 * @param {any} detail
 * @returns {string}
 */
function getItemValue(detail) {
  if (typeof detail === "string") return detail
  return detail?.value ?? detail?.text ?? detail?.label ?? ""
}

export class ComboboxControl extends Control {
  static plan = {
    tag: "ui-combobox",
    role: "combobox",
    tabIndex: -1,
    on: {
      focus: (e, target) => {
        if (target === document.activeElement) {
          target.inputEl.focus()
        }
      },
    },
  }

  /** @type {HTMLInputElement} */
  inputEl

  /** @type {ListboxComponent} */
  listboxEl

  #positionable
  #items = []
  #isOpen = false

  strict = false

  // MARK: items
  get items() {
    return this.#items
  }
  set items(value) {
    this.#items = value ?? []
    if (this.listboxEl) {
      this.listboxEl.content = this.#items
    }
  }

  // MARK: valueChanged
  valueChanged() {
    if (!this.inputEl) return
    this.inputEl.value = this.value
  }

  // MARK: #show
  #show() {
    if (this.#isOpen) return
    this.#isOpen = true

    this.listboxEl.hidden = false
    this.inputEl.setAttribute("aria-expanded", "true")

    this.#positionable?.destroy()
    this.#positionable = positionable(this.listboxEl, {
      signal: this.signal,
      preset: "popup",
      of: this,
    })
  }

  // MARK: showPicker
  showPicker() {
    this.#show()
    this.listboxEl.highlightFirst()
    this.#syncActiveDescendant()
  }

  // MARK: closePicker
  closePicker() {
    if (!this.#isOpen) return
    this.#isOpen = false

    this.listboxEl.hidden = true
    this.inputEl.setAttribute("aria-expanded", "false")
    this.inputEl.removeAttribute("aria-activedescendant")

    this.#positionable?.destroy()
    this.#positionable = undefined
  }

  // MARK: #syncActiveDescendant
  #syncActiveDescendant() {
    const item = this.listboxEl.currentItem
    if (item) {
      if (!item.id) {
        item.id = `${this.listboxEl.id}-opt-${this.listboxEl.current}`
      }
      this.inputEl.setAttribute("aria-activedescendant", item.id)
    } else {
      this.inputEl.removeAttribute("aria-activedescendant")
    }
  }

  // MARK: #pick
  #pick() {
    const entry = this.listboxEl.items[this.listboxEl.current]
    if (!entry) return

    this.listboxEl.pick(this.listboxEl.current)

    const value = getItemValue(entry.item ?? entry)
    this.setValue(value)
    this.inputEl.value = value
    this.closePicker()
  }

  // MARK: render
  render({ content }) {
    if (content) this.items = content

    return [
      {
        tag: "input.clear",
        role: "none",
        autocomplete: "off",
        value: this.value,
        created: (el) => {
          this.inputEl = el
        },
      },
      {
        tag: "button.addon",
        picto: "down",
        tabIndex: -1,
        on: {
          "disrupt": true,
          "pointerdown || Enter || Space": () => this.showPicker(),
        },
      },
      {
        tag: "ui-listbox",
        captureKeydown: false,
        fuzzy: true,
        content: this.#items,
        created: (el) => {
          this.listboxEl = /** @type {ListboxComponent} */ (el)
        },
      },
    ]
  }

  // MARK: created
  created() {
    const { signal } = this

    // Set initial ARIA state and hide listbox after it renders
    this.inputEl.setAttribute("aria-expanded", "false")
    this.listboxEl.ready.then(() => {
      this.inputEl.setAttribute("aria-controls", this.listboxEl.id)
      if (!this.#isOpen) this.listboxEl.hidden = true
      this.showPicker()
    })

    on(
      this.inputEl,
      { signal },
      {
        input: () => {
          if (!this.strict) {
            this.setValue(this.inputEl.value, { fromInput: true })
          }
          // Open before filtering so virtualizable has dimensions
          this.#show()
          this.listboxEl.search = this.inputEl.value
          this.#syncActiveDescendant()
        },
        focusout: () => {
          if (!this.strict) return
          const inputVal = this.inputEl.value
          const match = this.#items.some((item) => {
            const val = typeof item === "string" ? item : item?.value
            return val === inputVal
          })
          if (!match) {
            this.inputEl.value = this.value
          }
        },
      },
      {
        prevent: true,
        ArrowDown: () => {
          if (this.#isOpen) {
            this.listboxEl.highlightNext()
            this.#syncActiveDescendant()
          } else {
            this.showPicker()
          }
        },
        ArrowUp: () => {
          if (this.#isOpen) {
            this.listboxEl.highlightPrev()
            this.#syncActiveDescendant()
          }
        },
        Enter: () => {
          if (this.#isOpen) this.#pick()
        },
        Escape: () => {
          if (this.#isOpen) this.closePicker()
        },
      },
    )

    // Handle click on listbox items
    on(
      this.listboxEl,
      { signal },
      {
        selector: ".ui-menu__menuitem",
        pointerup: () => {
          this.#pick()
        },
      },
    )
  }
}

export const combobox = Control.define(ComboboxControl)
