/* eslint-disable max-depth */
/* eslint-disable complexity */
function ensureArrayBuffer(buf) {
  if (buf) {
    if (buf instanceof ArrayBuffer) return buf
    if (buf instanceof Uint8Array) return buf.buffer
  }
  throw new Error("Unsupported buffer type, need ArrayBuffer or Uint8Array")
}

// MARK: MIDIEvents
// Read and edit events from various sources (ArrayBuffer, Stream)
function MIDIEvents() {
  throw new Error("MIDIEvents function not intended to be run.")
}

// Static constants
// Event types
MIDIEvents.EVENT_META = 0xff
MIDIEvents.EVENT_SYSEX = 0xf0
MIDIEvents.EVENT_DIVSYSEX = 0xf7
MIDIEvents.EVENT_MIDI = 0x8
// Meta event types
MIDIEvents.EVENT_META_SEQUENCE_NUMBER = 0x00
MIDIEvents.EVENT_META_TEXT = 0x01
MIDIEvents.EVENT_META_COPYRIGHT_NOTICE = 0x02
MIDIEvents.EVENT_META_TRACK_NAME = 0x03
MIDIEvents.EVENT_META_INSTRUMENT_NAME = 0x04
MIDIEvents.EVENT_META_LYRICS = 0x05
MIDIEvents.EVENT_META_MARKER = 0x06
MIDIEvents.EVENT_META_CUE_POINT = 0x07
MIDIEvents.EVENT_META_MIDI_CHANNEL_PREFIX = 0x20
MIDIEvents.EVENT_META_END_OF_TRACK = 0x2f
MIDIEvents.EVENT_META_SET_TEMPO = 0x51
MIDIEvents.EVENT_META_SMTPE_OFFSET = 0x54
MIDIEvents.EVENT_META_TIME_SIGNATURE = 0x58
MIDIEvents.EVENT_META_KEY_SIGNATURE = 0x59
MIDIEvents.EVENT_META_SEQUENCER_SPECIFIC = 0x7f
// MIDI event types
MIDIEvents.EVENT_MIDI_NOTE_OFF = 0x8
MIDIEvents.EVENT_MIDI_NOTE_ON = 0x9
MIDIEvents.EVENT_MIDI_NOTE_AFTERTOUCH = 0xa
MIDIEvents.EVENT_MIDI_CONTROLLER = 0xb
MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE = 0xc
MIDIEvents.EVENT_MIDI_CHANNEL_AFTERTOUCH = 0xd
MIDIEvents.EVENT_MIDI_PITCH_BEND = 0xe
// MIDI event sizes
MIDIEvents.MIDI_1PARAM_EVENTS = [
  MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE,
  MIDIEvents.EVENT_MIDI_CHANNEL_AFTERTOUCH,
]
MIDIEvents.MIDI_2PARAMS_EVENTS = [
  MIDIEvents.EVENT_MIDI_NOTE_OFF,
  MIDIEvents.EVENT_MIDI_NOTE_ON,
  MIDIEvents.EVENT_MIDI_NOTE_AFTERTOUCH,
  MIDIEvents.EVENT_MIDI_CONTROLLER,
  MIDIEvents.EVENT_MIDI_PITCH_BEND,
]

// Create an event stream parser
MIDIEvents.createParser = function (stream, startAt, strictMode) {
  // Private vars
  // Common vars
  let eventTypeByte
  let event
  // MIDI events vars
  let MIDIEventType
  let MIDIEventChannel
  let MIDIEventParam1

  // Wrap DataView into a data stream
  if (stream instanceof DataView) {
    stream = {
      position: startAt || 0,
      buffer: stream,
      readUint8() {
        return this.buffer.getUint8(this.position++)
      },
      readUint16() {
        const v = this.buffer.getUint16(this.position)
        this.position += 2
        return v
      },
      readUint32() {
        const v = this.buffer.getUint16(this.position)
        this.position += 2
        return v
      },
      readVarInt() {
        let v = 0
        let i = 0
        let b
        while (i++ < 4) {
          b = this.readUint8()

          if (b & 0x80) {
            v += b & 0x7f
            v <<= 7
          } else {
            return v + b
          }
        }
        throw new Error(
          "0x" +
            this.position.toString(16) +
            ":" +
            " Variable integer length cannot exceed 4 bytes",
        )
      },
      readBytes(length) {
        const bytes = []

        for (; length > 0; length--) {
          bytes.push(this.readUint8())
        }
        return bytes
      },
      pos() {
        return "0x" + (this.buffer.byteOffset + this.position).toString(16)
      },
      index() {
        return this.buffer.byteOffset + this.position
      },
      end() {
        return this.position === this.buffer.byteLength
      },
    }
    startAt = 0
  }

  // Consume stream till not at start index
  if (startAt > 0) while (startAt--) stream.readUint8()

  // creating the parser object
  return {
    next() {
      if (stream.end()) return null

      // TODO: add types
      event = /** @type {any} */ ({})

      event.index = stream.index() // Memoize the event index
      event.delta = stream.readVarInt() // Read the delta time

      // Read the eventTypeByte
      eventTypeByte = stream.readUint8()
      if ((eventTypeByte & 0xf0) === 0xf0) {
        if (eventTypeByte === MIDIEvents.EVENT_META) {
          event.type = MIDIEvents.EVENT_META
          event.subtype = stream.readUint8()
          event.length = stream.readVarInt()
          switch (event.subtype) {
            case MIDIEvents.EVENT_META_SEQUENCE_NUMBER:
              if (strictMode && event.length !== 2) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              event.msb = stream.readUint8()
              event.lsb = stream.readUint8()
              return event
            case MIDIEvents.EVENT_META_TEXT:
            case MIDIEvents.EVENT_META_COPYRIGHT_NOTICE:
            case MIDIEvents.EVENT_META_TRACK_NAME:
            case MIDIEvents.EVENT_META_INSTRUMENT_NAME:
            case MIDIEvents.EVENT_META_LYRICS:
            case MIDIEvents.EVENT_META_MARKER:
            case MIDIEvents.EVENT_META_CUE_POINT:
              event.data = stream.readBytes(event.length)
              return event
            case MIDIEvents.EVENT_META_MIDI_CHANNEL_PREFIX:
              if (strictMode && event.length !== 1) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              event.prefix = stream.readUint8()
              return event
            case MIDIEvents.EVENT_META_END_OF_TRACK:
              if (strictMode && event.length > 0) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              return event
            case MIDIEvents.EVENT_META_SET_TEMPO:
              if (strictMode && event.length !== 3) {
                throw new Error(
                  `${stream.pos()} Tempo meta event length must be 3.`,
                )
              }
              event.tempo =
                (stream.readUint8() << 16) +
                (stream.readUint8() << 8) +
                stream.readUint8()
              event.tempoBPM = 60_000_000 / event.tempo
              return event
            case MIDIEvents.EVENT_META_SMTPE_OFFSET:
              if (strictMode && event.length !== 5) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              event.hour = stream.readUint8()
              if (strictMode && event.hour > 23) {
                throw new Error(
                  `${stream.pos()} SMTPE offset hour value must be part of 0-23.`,
                )
              }
              event.minutes = stream.readUint8()
              if (strictMode && event.minutes > 59) {
                throw new Error(
                  `${stream.pos()} SMTPE offset minutes value must be part of 0-59.`,
                )
              }
              event.seconds = stream.readUint8()
              if (strictMode && event.seconds > 59) {
                throw new Error(
                  `${stream.pos()} SMTPE offset seconds value must be part of 0-59.`,
                )
              }
              event.frames = stream.readUint8()
              if (strictMode && event.frames > 30) {
                throw new Error(
                  `${stream.pos()} SMTPE offset frames value must be part of 0-30.`,
                )
              }
              event.subframes = stream.readUint8()
              if (strictMode && event.subframes > 99) {
                throw new Error(
                  `${stream.pos()} SMTPE offset subframes value must be part of 0-99.`,
                )
              }
              return event
            case MIDIEvents.EVENT_META_KEY_SIGNATURE:
              if (strictMode && event.length !== 2) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              event.key = stream.readUint8()
              if (strictMode && (event.key < -7 || event.key > 7)) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              event.scale = stream.readUint8()
              if (strictMode && event.scale !== 0 && event.scale !== 1) {
                throw new Error(
                  `${stream.pos()} Key signature scale value must be 0 or 1.`,
                )
              }
              return event
            case MIDIEvents.EVENT_META_TIME_SIGNATURE:
              if (strictMode && event.length !== 4) {
                throw new Error(`${stream.pos()} Bad metaevent length.`)
              }
              event.data = stream.readBytes(event.length)
              event.param1 = event.data[0]
              event.param2 = event.data[1]
              event.param3 = event.data[2]
              event.param4 = event.data[3]
              return event
            case MIDIEvents.EVENT_META_SEQUENCER_SPECIFIC:
              event.data = stream.readBytes(event.length)
              return event
            default:
              if (strictMode) {
                throw new Error(
                  `${stream.pos()} Unknown meta event type (${event.subtype.toString(16)}).`,
                )
              }
              event.data = stream.readBytes(event.length)
              return event
          }
          // System events
        } else if (
          eventTypeByte === MIDIEvents.EVENT_SYSEX ||
          eventTypeByte === MIDIEvents.EVENT_DIVSYSEX
        ) {
          event.type = eventTypeByte
          event.length = stream.readVarInt()
          event.data = stream.readBytes(event.length)
          return event
          // Unknown event, assuming it's system like event
        } else {
          if (strictMode) {
            throw new Error(
              `${stream.pos()} Unknown event type ${eventTypeByte.toString(16)}, Delta: ${event.delta}.`,
            )
          }
          event.type = eventTypeByte
          event.badsubtype = stream.readVarInt()
          event.length = stream.readUint8()
          event.data = stream.readBytes(event.length)
          return event
        }
        // MIDI eventsdestination[index++]
      } else {
        // running status
        if ((eventTypeByte & 0x80) === 0) {
          if (!MIDIEventType) {
            throw new Error(
              `${stream.pos()} Running status without previous event`,
            )
          }
          MIDIEventParam1 = eventTypeByte
        } else {
          MIDIEventType = eventTypeByte >> 4
          MIDIEventChannel = eventTypeByte & 0x0f
          MIDIEventParam1 = stream.readUint8()
        }
        event.type = MIDIEvents.EVENT_MIDI
        event.subtype = MIDIEventType
        event.channel = MIDIEventChannel
        event.param1 = MIDIEventParam1
        switch (MIDIEventType) {
          case MIDIEvents.EVENT_MIDI_NOTE_OFF:
            event.param2 = stream.readUint8()
            return event
          case MIDIEvents.EVENT_MIDI_NOTE_ON:
            event.param2 = stream.readUint8()

            // If velocity is 0, it's a note off event in fact
            if (!event.param2) {
              event.subtype = MIDIEvents.EVENT_MIDI_NOTE_OFF
              event.param2 = 127 // Find a standard telling what to do here
            }
            return event
          case MIDIEvents.EVENT_MIDI_NOTE_AFTERTOUCH:
            event.param2 = stream.readUint8()
            return event
          case MIDIEvents.EVENT_MIDI_CONTROLLER:
            event.param2 = stream.readUint8()
            return event
          case MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE:
            return event
          case MIDIEvents.EVENT_MIDI_CHANNEL_AFTERTOUCH:
            return event
          case MIDIEvents.EVENT_MIDI_PITCH_BEND:
            event.param2 = stream.readUint8()
            return event
          default:
            if (strictMode) {
              throw new Error(
                `${stream.pos()} Unknown MIDI event type (${MIDIEventType.toString(16)}).`,
              )
            }
            return event
        }
      }
    },
  }
}

// Return the buffer length needed to encode the given events
MIDIEvents.writeToTrack = function (events, destination, strictMode) {
  let index = 0
  let i
  let j
  let k
  let l

  // Converting each event to binary MIDI datas
  for (i = 0, j = events.length; i < j; i++) {
    // Writing delta value
    if (events[i].delta >>> 28) {
      throw new Error(
        `Event #${i}: Maximum delta time value reached (${events[i].delta}/134217728 max)`,
      )
    }
    if (events[i].delta >>> 21) {
      destination[index++] = ((events[i].delta >>> 21) & 0x7f) | 0x80
    }
    if (events[i].delta >>> 14) {
      destination[index++] = ((events[i].delta >>> 14) & 0x7f) | 0x80
    }
    if (events[i].delta >>> 7) {
      destination[index++] = ((events[i].delta >>> 7) & 0x7f) | 0x80
    }
    destination[index++] = events[i].delta & 0x7f
    // MIDI Events encoding
    if (events[i].type === MIDIEvents.EVENT_MIDI) {
      // Adding the byte of subtype + channel
      destination[index++] = (events[i].subtype << 4) + events[i].channel
      // Adding the byte of the first params
      destination[index++] = events[i].param1
      // Adding a byte for the optionnal second param
      if (MIDIEvents.MIDI_2PARAMS_EVENTS.includes(events[i].subtype)) {
        destination[index++] = events[i].param2
      }
      // META / SYSEX events encoding
    } else {
      // Adding the event type byte
      destination[index++] = events[i].type
      // Adding the META event subtype byte
      if (events[i].type === MIDIEvents.EVENT_META) {
        destination[index++] = events[i].subtype
      }
      // Writing the event length bytes
      if (events[i].length >>> 28) {
        throw new Error(
          `Event #${i}: Maximum length reached (${events[i].length}/134217728 max)`,
        )
      }
      if (events[i].length >>> 21) {
        destination[index++] = ((events[i].length >>> 21) & 0x7f) | 0x80
      }
      if (events[i].length >>> 14) {
        destination[index++] = ((events[i].length >>> 14) & 0x7f) | 0x80
      }
      if (events[i].length >>> 7) {
        destination[index++] = ((events[i].length >>> 7) & 0x7f) | 0x80
      }
      destination[index++] = events[i].length & 0x7f
      if (events[i].type === MIDIEvents.EVENT_META) {
        switch (events[i].subtype) {
          case MIDIEvents.EVENT_META_SEQUENCE_NUMBER:
            destination[index++] = events[i].msb
            destination[index++] = events[i].lsb
            break
          case MIDIEvents.EVENT_META_TEXT:
          case MIDIEvents.EVENT_META_COPYRIGHT_NOTICE:
          case MIDIEvents.EVENT_META_TRACK_NAME:
          case MIDIEvents.EVENT_META_INSTRUMENT_NAME:
          case MIDIEvents.EVENT_META_LYRICS:
          case MIDIEvents.EVENT_META_MARKER:
          case MIDIEvents.EVENT_META_CUE_POINT:
            for (k = 0, l = events[i].length; k < l; k++) {
              destination[index++] = events[i].data[k]
            }
            break
          case MIDIEvents.EVENT_META_MIDI_CHANNEL_PREFIX:
            destination[index++] = events[i].prefix
            break
          case MIDIEvents.EVENT_META_END_OF_TRACK:
            break
          case MIDIEvents.EVENT_META_SET_TEMPO:
            destination[index++] = events[i].tempo >> 16
            destination[index++] = (events[i].tempo >> 8) & 0xff
            destination[index++] = events[i].tempo & 0xff
            break
          case MIDIEvents.EVENT_META_SMTPE_OFFSET:
            if (strictMode && events[i].hour > 23) {
              throw new Error(
                `Event #${i}: SMTPE offset hour value must be part of 0-23.`,
              )
            }
            destination[index++] = events[i].hour
            if (strictMode && events[i].minutes > 59) {
              throw new Error(
                `Event #${i}: SMTPE offset minutes value must be part of 0-59.`,
              )
            }
            destination[index++] = events[i].minutes
            if (strictMode && events[i].seconds > 59) {
              throw new Error(
                `Event #${i}: SMTPE offset seconds value must be part of 0-59.`,
              )
            }
            destination[index++] = events[i].seconds
            if (strictMode && events[i].frames > 30) {
              throw new Error(
                `Event #${i}: SMTPE offset frames amount must be part of 0-30.`,
              )
            }
            destination[index++] = events[i].frames
            if (strictMode && events[i].subframes > 99) {
              throw new Error(
                `Event #${i}: SMTPE offset subframes amount must be part of 0-99.`,
              )
            }
            destination[index++] = events[i].subframes
            break
          case MIDIEvents.EVENT_META_KEY_SIGNATURE:
            if (
              typeof events[i].key !== "number" ||
              events[i].key < -7 ||
              events[i].scale > 7
            ) {
              throw new Error(
                `Event #${i}:The key signature key must be between -7 and 7`,
              )
            }
            if (
              typeof events[i].scale !== "number" ||
              events[i].scale < 0 ||
              events[i].scale > 1
            ) {
              throw new Error(
                `Event #${i}: The key signature scale must be 0 or 1`,
              )
            }
            destination[index++] = events[i].key
            destination[index++] = events[i].scale
            break

          // // Not implemented
          // case MIDIEvents.EVENT_META_TIME_SIGNATURE:
          // case MIDIEvents.EVENT_META_SEQUENCER_SPECIFIC:
          default:
            for (k = 0, l = events[i].length; k < l; k++) {
              destination[index++] = events[i].data[k]
            }
            break
        }
        // Adding bytes corresponding to the sysex event datas
      } else {
        for (k = 0, l = events[i].length; k < l; k++) {
          destination[index++] = events[i].data[k]
        }
      }
    }
  }
}

// Return the buffer length needed to encode the given events
MIDIEvents.getRequiredBufferLength = function (events) {
  let bufferLength = 0
  let i = 0
  let j

  // Calculating the track size by adding events lengths
  for (i = 0, j = events.length; i < j; i++) {
    // Computing necessary bytes to encode the delta value
    bufferLength +=
      events[i].delta >>> 21
        ? 4
        : events[i].delta >>> 14
          ? 3
          : events[i].delta >>> 7
            ? 2
            : 1
    // MIDI Events have various fixed lengths
    if (events[i].type === MIDIEvents.EVENT_MIDI) {
      // Adding a byte for subtype + channel
      bufferLength++
      // Adding a byte for the first params
      bufferLength++
      // Adding a byte for the optionnal second param
      if (MIDIEvents.MIDI_2PARAMS_EVENTS.includes(events[i].subtype)) {
        bufferLength++
      }
      // META / SYSEX events lengths are self defined
    } else {
      // Adding a byte for the event type
      bufferLength++
      // Adding a byte for META events subtype
      if (events[i].type === MIDIEvents.EVENT_META) {
        bufferLength++
      }
      // Adding necessary bytes to encode the length
      bufferLength +=
        events[i].length >>> 21
          ? 4
          : events[i].length >>> 14
            ? 3
            : events[i].length >>> 7
              ? 2
              : 1
      // Adding bytes corresponding to the event length
      bufferLength += events[i].length
    }
  }
  return bufferLength
}

// MARK: MIDIFileHeader
// Read and edit a MIDI header chunk in a given ArrayBuffer
class MIDIFileHeader {
  static HEADER_LENGTH = 14
  static FRAMES_PER_SECONDS = 1
  static TICKS_PER_BEAT = 2

  constructor(buffer) {
    // No buffer creating him
    if (buffer) {
      if (!(buffer instanceof ArrayBuffer)) {
        throw new TypeError("Invalid buffer received.")
      }
      this.datas = new DataView(buffer, 0, MIDIFileHeader.HEADER_LENGTH)
      // Reading MIDI header chunk
      if (
        !(
          String.fromCharCode(this.datas.getUint8(0)) === "M" &&
          String.fromCharCode(this.datas.getUint8(1)) === "T" &&
          String.fromCharCode(this.datas.getUint8(2)) === "h" &&
          String.fromCharCode(this.datas.getUint8(3)) === "d"
        )
      ) {
        throw new Error("Invalid MIDIFileHeader : MThd prefix not found")
      }
      // Reading chunk length
      if (this.datas.getUint32(4) !== 6) {
        throw new Error("Invalid MIDIFileHeader : Chunk length must be 6")
      }
    } else {
      const a = new Uint8Array(MIDIFileHeader.HEADER_LENGTH)
      // Adding the header id (MThd)
      a[0] = 0x4d
      a[1] = 0x54
      a[2] = 0x68
      a[3] = 0x64
      // Adding the header chunk size
      a[4] = 0x00
      a[5] = 0x00
      a[6] = 0x00
      a[7] = 0x06
      // Adding the file format (1 here cause it's the most commonly used)
      a[8] = 0x00
      a[9] = 0x01
      // Adding the track count (1 cause it's a new file)
      a[10] = 0x00
      a[11] = 0x01
      // Adding the time division (192 ticks per beat)
      a[12] = 0x00
      a[13] = 0xc0
      // saving the buffer
      this.datas = new DataView(a.buffer, 0, MIDIFileHeader.HEADER_LENGTH)
      // Parsing the given buffer
    }
  }

  // MIDI file format
  getFormat() {
    const format = this.datas.getUint16(8)
    if (format !== 0 && format !== 1 && format !== 2) {
      throw new Error(
        "Invalid MIDI file : MIDI format (" +
          format +
          ")," +
          " format can be 0, 1 or 2 only.",
      )
    }
    return format
  }
  setFormat(format) {
    if (format !== 0 && format !== 1 && format !== 2) {
      throw new Error(
        "Invalid MIDI format given (" +
          format +
          ")," +
          " format can be 0, 1 or 2 only.",
      )
    }
    this.datas.setUint16(8, format)
  }

  // Number of tracks
  getTracksCount() {
    return this.datas.getUint16(10)
  }
  setTracksCount(n) {
    return this.datas.setUint16(10, n)
  }

  // Tick compute
  getTickResolution(tempo) {
    // Frames per seconds
    if (this.datas.getUint16(12) & 32_768) {
      return 1_000_000 / (this.getSMPTEFrames() * this.getTicksPerFrame()) // Ticks per beat
    }
    // Default MIDI tempo is 120bpm, 500ms per beat
    tempo = tempo || 500_000
    return tempo / this.getTicksPerBeat()
  }

  // Time division type
  getTimeDivision() {
    if (this.datas.getUint16(12) & 32_768) {
      return MIDIFileHeader.FRAMES_PER_SECONDS
    }
    return MIDIFileHeader.TICKS_PER_BEAT
  }

  // Ticks per beat
  getTicksPerBeat() {
    const divisionWord = this.datas.getUint16(12)
    if (divisionWord & 32_768) {
      throw new Error("Time division is not expressed as ticks per beat.")
    }
    return divisionWord
  }
  setTicksPerBeat(ticksPerBeat) {
    this.datas.setUint16(12, ticksPerBeat & 32_767)
  }

  // Frames per seconds
  getSMPTEFrames() {
    const divisionWord = this.datas.getUint16(12)
    if (!(divisionWord & 32_768)) {
      throw new Error("Time division is not expressed as frames per seconds.")
    }
    const smpteFrames = divisionWord & 32_512
    if (![24, 25, 29, 30].includes(smpteFrames)) {
      throw new Error("Invalid SMPTE frames value (" + smpteFrames + ").")
    }
    return smpteFrames === 29 ? 29.97 : smpteFrames
  }
  getTicksPerFrame() {
    const divisionWord = this.datas.getUint16(12)
    if (!(divisionWord & 32_768)) {
      throw new Error("Time division is not expressed as frames per seconds.")
    }
    return divisionWord & 255
  }
  setSMTPEDivision(smpteFrames, ticksPerFrame) {
    if (smpteFrames === 29.97) smpteFrames = 29
    if (![24, 25, 29, 30].includes(smpteFrames)) {
      throw new Error("Invalid SMPTE frames value given (" + smpteFrames + ").")
    }
    if (ticksPerFrame < 0 || ticksPerFrame > 0xff) {
      throw new Error(
        "Invalid ticks per frame value given (" + smpteFrames + ").",
      )
    }
    this.datas.setUint8(12, 0x80 | smpteFrames)
    this.datas.setUint8(13, ticksPerFrame)
  }
}

// MARK: MIDIFileTrack
// Read and edit a MIDI track chunk in a given ArrayBuffer
class MIDIFileTrack {
  static HDR_LENGTH = 8

  constructor(buffer, start) {
    let a
    let trackLength

    // no buffer, creating him
    if (buffer) {
      if (!(buffer instanceof ArrayBuffer)) {
        throw new TypeError("Invalid buffer received.")
      }
      // Buffer length must size at least like an  empty track (8+3bytes)
      if (buffer.byteLength - start < 12) {
        throw new Error(
          `Invalid MIDIFileTrack (0x${start.toString(16)}) : Buffer length must size at least 12bytes`,
        )
      }
      // Creating a temporary view to read the track header
      this.datas = new DataView(buffer, start, MIDIFileTrack.HDR_LENGTH)
      // Reading MIDI track header chunk
      if (
        !(
          String.fromCharCode(this.datas.getUint8(0)) === "M" &&
          String.fromCharCode(this.datas.getUint8(1)) === "T" &&
          String.fromCharCode(this.datas.getUint8(2)) === "r" &&
          String.fromCharCode(this.datas.getUint8(3)) === "k"
        )
      ) {
        throw new Error(
          `Invalid MIDIFileTrack (0x${start.toString(16)}) : MTrk prefix not found`,
        )
      }
      // Reading the track length
      trackLength = this.getTrackLength()
      if (buffer.byteLength - start < trackLength) {
        throw new Error(
          `Invalid MIDIFileTrack (0x${start.toString(
            16,
          )}) : The track size exceed the buffer length.`,
        )
      }
      // Creating the final DataView
      this.datas = new DataView(
        buffer,
        start,
        MIDIFileTrack.HDR_LENGTH + trackLength,
      )
      // Trying to find the end of track event
      if (
        !(
          this.datas.getUint8(MIDIFileTrack.HDR_LENGTH + (trackLength - 3)) ===
            0xff &&
          this.datas.getUint8(MIDIFileTrack.HDR_LENGTH + (trackLength - 2)) ===
            0x2f &&
          this.datas.getUint8(MIDIFileTrack.HDR_LENGTH + (trackLength - 1)) ===
            0x00
        )
      ) {
        throw new Error(
          `Invalid MIDIFileTrack (0x${start.toString(
            16,
          )}) : No track end event found at the expected index (${(
            MIDIFileTrack.HDR_LENGTH +
            (trackLength - 1)
          ).toString(16)}).`,
        )
      }
    } else {
      a = new Uint8Array(12)
      // Adding the empty track header (MTrk)
      a[0] = 0x4d
      a[1] = 0x54
      a[2] = 0x72
      a[3] = 0x6b
      // Adding the empty track size (4)
      a[4] = 0x00
      a[5] = 0x00
      a[6] = 0x00
      a[7] = 0x04
      // Adding the track end event
      a[8] = 0x00
      a[9] = 0xff
      a[10] = 0x2f
      a[11] = 0x00
      // Saving the buffer
      this.datas = new DataView(a.buffer, 0, MIDIFileTrack.HDR_LENGTH + 4)
      // parsing the given buffer
    }
  }

  // Track length
  getTrackLength() {
    return this.datas.getUint32(4)
  }
  setTrackLength(trackLength) {
    return this.datas.setUint32(4, trackLength)
  }

  // Read track contents
  getTrackContent() {
    return new DataView(
      this.datas.buffer,
      this.datas.byteOffset + MIDIFileTrack.HDR_LENGTH,
      this.datas.byteLength - MIDIFileTrack.HDR_LENGTH,
    )
  }

  // Set track content
  setTrackContent(dataView) {
    // Calculating the track length
    const trackLength = dataView.byteLength - dataView.byteOffset

    // Track length must size at least like an  empty track (4bytes)
    if (trackLength < 4) {
      throw new Error("Invalid track length, must size at least 4bytes")
    }
    this.datas = new DataView(
      new Uint8Array(MIDIFileTrack.HDR_LENGTH + trackLength).buffer,
    )
    // Adding the track header (MTrk)
    this.datas.setUint8(0, 0x4d) // M
    this.datas.setUint8(1, 0x54) // T
    this.datas.setUint8(2, 0x72) // r
    this.datas.setUint8(3, 0x6b) // k

    // Adding the track size
    this.datas.setUint32(4, trackLength)
    // Copying the content
    const origin = new Uint8Array(
      dataView.buffer,
      dataView.byteOffset,
      dataView.byteLength,
    )
    const destination = new Uint8Array(
      this.datas.buffer,
      MIDIFileTrack.HDR_LENGTH,
      trackLength,
    )

    for (let i = 0, l = origin.length; i < l; i++) destination[i] = origin[i]
  }
}

// MARK: MIDIFile
// Read (and soon edit) a MIDI file in a given ArrayBuffer
class MIDIFile {
  static Header = MIDIFileHeader
  static Track = MIDIFileTrack

  constructor(buffer, strictMode) {
    let track
    let curIndex

    // If not buffer given, creating a new MIDI file
    if (buffer) {
      buffer = ensureArrayBuffer(buffer)
      // Minimum MIDI file size is a headerChunk size (14bytes)
      // and an empty track (8+3bytes)
      if (buffer.byteLength < 25) {
        throw new Error(
          `A buffer of a valid MIDI file must have, at least, a size of 25bytes.`,
        )
      }
      // Reading header
      this.header = new MIDIFileHeader(buffer)
      this.tracks = []
      curIndex = MIDIFileHeader.HEADER_LENGTH
      // Reading tracks
      for (let i = 0, l = this.header.getTracksCount(); i < l; i++) {
        // Testing the buffer length
        if (strictMode && curIndex >= buffer.byteLength - 1) {
          throw new Error(
            `Couldn't find datas corresponding to the track #${i}.`,
          )
        }
        // Creating the track object
        track = new MIDIFileTrack(buffer, curIndex)
        this.tracks.push(track)
        // Updating index to the track end
        curIndex += track.getTrackLength() + 8
      }
      // Testing integrity : curIndex should be at the end of the buffer
      if (strictMode && curIndex !== buffer.byteLength) {
        throw new Error("It seems that the buffer contains too much datas.")
      }
    } else {
      // Creating the content
      this.header = new MIDIFileHeader()
      this.tracks = [new MIDIFileTrack()]
      // if a buffer is provided, parsing him
    }
  }

  startNote(event, song) {
    const track = this.takeTrack(event.channel, song)
    track.notes.push({
      when: event.playTime / 1000,
      pitch: event.param1,
      duration: 1e-7,
      slides: [],
    })
  }

  closeNote(event, song) {
    const track = this.takeTrack(event.channel, song)
    for (let i = 0; i < track.notes.length; i++) {
      if (
        track.notes[i].duration === 1e-7 &&
        track.notes[i].pitch === event.param1 &&
        track.notes[i].when < event.playTime / 1000
      ) {
        track.notes[i].duration = event.playTime / 1000 - track.notes[i].when
        break
      }
    }
  }

  addSlide(event, song, pitchBendRange) {
    const track = this.takeTrack(event.channel, song)
    for (let i = 0; i < track.notes.length; i++) {
      if (
        track.notes[i].duration === 1e-7 &&
        track.notes[i].when < event.playTime / 1000
      ) {
        // if (Math.abs(track.notes[i].shift) < Math.abs(event.param2 - 64) / 6) {
        // track.notes[i].shift = (event.param2 - 64) / 6;
        // console.log(event.param2-64);
        // }
        track.notes[i].slides.push({
          // pitch: track.notes[i].pitch + (event.param2 - 64) / 6,
          delta: ((event.param2 - 64) / 64) * pitchBendRange,
          when: event.playTime / 1000 - track.notes[i].when,
        })
      }
    }
  }

  startDrum(event, song) {
    const beat = this.takeBeat(event.param1, song)
    beat.notes.push({ when: event.playTime / 1000 })
  }

  takeTrack(n, song) {
    for (let i = 0; i < song.tracks.length; i++) {
      if (song.tracks[i].n === n) return song.tracks[i]
    }

    const track = {
      n,
      notes: [],
      volume: 1,
      program: 0,
    }
    song.tracks.push(track)
    return track
  }

  takeBeat(n, song) {
    for (let i = 0; i < song.beats.length; i++) {
      if (song.beats[i].n === n) return song.beats[i]
    }

    const beat = {
      n,
      notes: [],
      volume: 1,
    }
    song.beats.push(beat)
    return beat
  }

  parseSong() {
    const song = {
      duration: 0,
      tracks: [],
      beats: [],
    }

    const events = this.getMidiEvents()

    // To set the pitch-bend range, three to four consecutive EVENT_MIDI_CONTROLLER messages must have consistent contents.
    let expectedPitchBendRangeMessageNumber = 1 // counts which pitch-bend range message can be expected next: number 1 (can be sent any time, except after pitch-bend range messages number 1 or 2), number 2 (required after number 1), number 3 (required after number 2), or number 4 (optional)
    let expectedPitchBendRangeChannel = null
    const pitchBendRange = new Array(16).fill(2) // Default pitch-bend range is 2 semitones.

    for (let i = 0; i < events.length; i++) {
      const expectedPitchBendRangeMessageNumberOld =
        expectedPitchBendRangeMessageNumber
      // console.debug('		next',events[i]);
      if (song.duration < events[i].playTime / 1000) {
        song.duration = events[i].playTime / 1000
      }
      if (events[i].subtype === MIDIEvents.EVENT_MIDI_NOTE_ON) {
        if (events[i].channel === 9) {
          if (events[i].param1 >= 35 && events[i].param1 <= 81) {
            this.startDrum(events[i], song)
          } else {
            console.debug("wrong drum", events[i])
          }
        } else if (events[i].param1 >= 0 && events[i].param1 <= 127) {
          // console.debug('start', events[i].param1);
          this.startNote(events[i], song)
        } else {
          console.debug("wrong tone", events[i])
        }
      } else if (events[i].subtype === MIDIEvents.EVENT_MIDI_NOTE_OFF) {
        if (events[i].channel !== 9) {
          this.closeNote(events[i], song)
          // console.debug('close', events[i].param1);
        }
      } else if (events[i].subtype === MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE) {
        if (events[i].channel === 9) {
          console.debug("skip program for drums")
        } else {
          const track = this.takeTrack(events[i].channel, song)
          track.program = events[i].param1
        }
      } else if (events[i].subtype === MIDIEvents.EVENT_MIDI_CONTROLLER) {
        if (events[i].param1 === 7) {
          if (events[i].channel !== 9) {
            // TODO why not set loudness for drums?
            const track = this.takeTrack(events[i].channel, song)
            track.volume = events[i].param2 / 127 || 0.000_001
            // console.debug('volume', track.volume,'for',events[i].channel);
          }
        } else if (
          (expectedPitchBendRangeMessageNumber === 1 &&
            events[i].param1 === 0x65 &&
            events[i].param2 === 0x00) ||
          (expectedPitchBendRangeMessageNumber === 2 &&
            events[i].param1 === 0x64 &&
            events[i].param2 === 0x00) ||
          (expectedPitchBendRangeMessageNumber === 3 &&
            events[i].param1 === 0x06) ||
          (expectedPitchBendRangeMessageNumber === 4 &&
            events[i].param1 === 0x26)
        ) {
          if (
            expectedPitchBendRangeMessageNumber > 1 &&
            events[i].channel !== expectedPitchBendRangeChannel
          ) {
            // no error
            console.debug(
              "Unexpected channel number in non-first pitch-bend RANGE (SENSITIVITY) message. MIDI file might be corrupt.",
            )
          }
          expectedPitchBendRangeChannel = events[i].channel
          if (expectedPitchBendRangeMessageNumber === 3) {
            pitchBendRange[events[i].channel] = events[i].param2 // in semitones
            console.debug("pitchBendRange", pitchBendRange)
          }
          if (expectedPitchBendRangeMessageNumber === 4) {
            pitchBendRange[events[i].channel] += events[i].param2 / 100 // convert cents to semitones, add to semitones set in the previous MIDI message
            console.debug("pitchBendRange", pitchBendRange)
          }
          expectedPitchBendRangeMessageNumber++
          if (expectedPitchBendRangeMessageNumber === 5) {
            expectedPitchBendRangeMessageNumber = 1
          }
        } else {
          // console.debug('controller', events[i]);
        }
      } else if (events[i].subtype === MIDIEvents.EVENT_MIDI_PITCH_BEND) {
        // console.debug('	bend', events[i].channel, events[i].param1, events[i].param2);
        this.addSlide(events[i], song, pitchBendRange[events[i].channel])
      } else {
        console.debug("unknown", events[i].channel, events[i])
      }
      if (
        expectedPitchBendRangeMessageNumberOld ===
        expectedPitchBendRangeMessageNumber
      ) {
        // If the current message wasn't an expected pitch-bend range message
        if (
          expectedPitchBendRangeMessageNumberOld >= 2 &&
          expectedPitchBendRangeMessageNumberOld <= 3
        ) {
          // no error
          console.debug(
            "Pitch-bend RANGE (SENSITIVITY) messages ended prematurely. MIDI file might be corrupt.",
          )
        }
        if (expectedPitchBendRangeMessageNumberOld === 4) {
          // The fourth message is optional, so since it wasn't sent, the setting of the pitch-bend range is done, and we might expect the first pitch-bend range message some time in the future
          expectedPitchBendRangeMessageNumber = 1
        }
      }
    }
    return song
  }

  // Events reading helpers
  getEvents(type, subtype) {
    let events
    let event
    let playTime = 0
    const filteredEvents = []
    const format = this.header.getFormat()
    let tickResolution = this.header.getTickResolution()
    let i
    let j
    let trackParsers
    let smallestDelta

    // Reading events
    // if the read is sequential
    if (format !== 1 || this.tracks.length === 1) {
      for (i = 0, j = this.tracks.length; i < j; i++) {
        // reset playtime if format is 2
        playTime = format === 2 && playTime ? playTime : 0
        events = MIDIEvents.createParser(
          this.tracks[i].getTrackContent(),
          0,
          false,
        )
        // loooping through events
        event = events.next()
        while (event) {
          playTime += event.delta ? (event.delta * tickResolution) / 1000 : 0
          if (event.type === MIDIEvents.EVENT_META) {
            // tempo change events
            if (event.subtype === MIDIEvents.EVENT_META_SET_TEMPO) {
              tickResolution = this.header.getTickResolution(event.tempo)
            }
          }
          // push the asked events
          if (
            (!type || event.type === type) &&
            (!subtype || (event.subtype && event.subtype === subtype))
          ) {
            event.playTime = playTime
            filteredEvents.push(event)
          }
          event = events.next()
        }
      }
      // the read is concurrent
    } else {
      // TODO: add types
      trackParsers = /** @type {any[]} */ ([])
      smallestDelta = -1

      // Creating parsers
      for (i = 0, j = this.tracks.length; i < j; i++) {
        trackParsers[i] = {}
        trackParsers[i].parser = MIDIEvents.createParser(
          this.tracks[i].getTrackContent(),
          0,
          false,
        )
        trackParsers[i].curEvent = trackParsers[i].parser.next()
      }

      // Filling events
      do {
        smallestDelta = -1
        // finding the smallest event
        for (i = 0, j = trackParsers.length; i < j; i++) {
          if (trackParsers[i].curEvent) {
            if (
              smallestDelta === -1 ||
              trackParsers[i].curEvent.delta <
                trackParsers[smallestDelta].curEvent.delta
            ) {
              smallestDelta = i
            }
          }
        }
        if (smallestDelta !== -1) {
          // removing the delta of previous events
          for (i = 0, j = trackParsers.length; i < j; i++) {
            if (i !== smallestDelta && trackParsers[i].curEvent) {
              trackParsers[i].curEvent.delta -=
                trackParsers[smallestDelta].curEvent.delta
            }
          }
          // filling values
          event = trackParsers[smallestDelta].curEvent
          playTime += event.delta ? (event.delta * tickResolution) / 1000 : 0
          if (event.type === MIDIEvents.EVENT_META) {
            // tempo change events
            if (event.subtype === MIDIEvents.EVENT_META_SET_TEMPO) {
              tickResolution = this.header.getTickResolution(event.tempo)
            }
          }
          // push midi events
          if (
            (!type || event.type === type) &&
            (!subtype || (event.subtype && event.subtype === subtype))
          ) {
            event.playTime = playTime
            event.track = smallestDelta
            filteredEvents.push(event)
          }
          // getting next event
          trackParsers[smallestDelta].curEvent =
            trackParsers[smallestDelta].parser.next()
        }
      } while (smallestDelta !== -1)
    }
    return filteredEvents
  }

  getMidiEvents() {
    return this.getEvents(MIDIEvents.EVENT_MIDI)
  }

  getLyrics() {
    const events = this.getEvents(MIDIEvents.EVENT_META)
    let texts = []
    const lyrics = []
    let event
    let i
    let j

    for (i = 0, j = events.length; i < j; i++) {
      event = events[i]

      if (event.subtype === MIDIEvents.EVENT_META_LYRICS) {
        lyrics.push(event)
      } else if (event.subtype === MIDIEvents.EVENT_META_TEXT) {
        if (String.fromCharCode(event.data[0]) === "@") {
          // Ignore special texts
          if (String.fromCharCode(event.data[1]) === "T") {
            console.log("Title : " + event.text.slice(2))
          } else if (String.fromCharCode(event.data[1]) === "I") {
            console.log("Info : " + event.text.slice(2))
          } else if (String.fromCharCode(event.data[1]) === "L") {
            console.log("Lang : " + event.text.slice(2))
          }
        } else if (String.fromCharCode(...event.data).indexOf("words") === 0) {
          // karaoke text follows, remove all previous text
          texts.length = 0
          // console.log('Word marker found');
        } else if (event.playTime !== 0) {
          // Karaoke texts
          texts.push(event)
        }
      }
    }

    // Choosing the right lyrics
    if (lyrics.length > 2) {
      texts = lyrics
    } else if (texts.length === 0) {
      texts = []
    }

    // Convert texts and detect encoding
    try {
      const td = new TextDecoder()
      texts.forEach((event) => {
        // event.text = UTF8.getStringFromBytes(event.data, 0, event.length, true)
        event.text = td.decode(new Uint8Array(event.data).buffer)
      })
    } catch (err) {
      console.log(err)
      texts.forEach((event) => {
        event.text = event.data.map((c) => String.fromCharCode(c)).join("")
      })
    }
    return texts
  }

  // Basic events reading
  getTrackEvents(index) {
    if (index > this.tracks.length || index < 0) {
      throw new Error("Invalid track index (" + index + ")")
    }

    let event
    const events = []
    const parser = MIDIEvents.createParser(
      this.tracks[index].getTrackContent(),
      0,
      false,
    )
    event = parser.next()

    do {
      events.push(event)
      event = parser.next()
    } while (event)

    return events
  }

  // Basic events writting
  setTrackEvents(index, events) {
    if (index > this.tracks.length || index < 0) {
      throw new Error("Invalid track index (" + index + ")")
    }
    if (!events || events.length === 0) {
      throw new Error("A track must contain at least one event, none given.")
    }

    const bufferLength = MIDIEvents.getRequiredBufferLength(events)
    const destination = new Uint8Array(bufferLength)
    MIDIEvents.writeToTrack(events, destination)
    this.tracks[index].setTrackContent(destination)
  }

  // Remove a track
  deleteTrack(index) {
    if (index > this.tracks.length || index < 0) {
      throw new Error("Invalid track index (" + index + ")")
    }

    this.tracks.splice(index, 1)
    this.header.setTracksCount(this.tracks.length)
  }

  // Add a track
  addTrack(index) {
    if (index > this.tracks.length || index < 0) {
      throw new Error("Invalid track index (" + index + ")")
    }

    const track = new MIDIFileTrack()
    if (index === this.tracks.length) this.tracks.push(track)
    else this.tracks.splice(index, 0, track)

    this.header.setTracksCount(this.tracks.length)
  }

  // Retrieve the content in a buffer
  getContent() {
    let bufferLength
    let origin
    let i
    let j
    let k
    let l
    let m
    let n

    // Calculating the buffer content
    // - initialize with the header length
    bufferLength = MIDIFileHeader.HEADER_LENGTH
    // - add tracks length
    for (i = 0, j = this.tracks.length; i < j; i++) {
      bufferLength += this.tracks[i].getTrackLength() + 8
    }

    // Creating the destination buffer
    const destination = new Uint8Array(bufferLength)

    // Adding header
    origin = new Uint8Array(
      this.header.datas.buffer,
      this.header.datas.byteOffset,
      MIDIFileHeader.HEADER_LENGTH,
    )
    for (i = 0, j = MIDIFileHeader.HEADER_LENGTH; i < j; i++) {
      destination[i] = origin[i]
    }

    // Adding tracks
    for (k = 0, l = this.tracks.length; k < l; k++) {
      origin = new Uint8Array(
        this.tracks[k].datas.buffer,
        this.tracks[k].datas.byteOffset,
        this.tracks[k].datas.byteLength,
      )
      for (m = 0, n = this.tracks[k].datas.byteLength; m < n; m++) {
        destination[i++] = origin[m]
      }
    }

    return destination.buffer
  }
}

export { MIDIFile, MIDIFileHeader, MIDIFileTrack, MIDIEvents }
