// @ts-nocheck
import { ConfigFile } from "../ConfigFile.js"
import { fs } from "../../fs.js"
import { loadDesktopStyle } from "../loadDesktopStyle.js"
import { fileIndex } from "../../fileIndex.js"
// import { loadText } from "../../load/loadText.js"

const DEFAULTS = {
  activeUser: "windows93",
  users: ["windows93"],
}

export const DEFAULT_DIRECTORIES = {
  "/bookmarks": "[Desktop Entry]\nIcon=folder-bookmark\n",
  "/books": "[Desktop Entry]\nIcon=folder-books\n",
  "/config": "[Desktop Entry]\nIcon=folder-development\n",
  "/desktop": "[Desktop Entry]\nIcon=folder-desktop\n",
  "/documents": "[Desktop Entry]\nIcon=folder-documents\n",
  "/interface": "[Desktop Entry]\nIcon=folder-activities\n",
  "/models": "[Desktop Entry]\nIcon=folder-models\n",
  "/music": "[Desktop Entry]\nIcon=folder-music\n",
  "/pictures": "[Desktop Entry]\nIcon=folder-image\n",
  "/roms": "[Desktop Entry]\nIcon=folder-games\n",
  "/sounds": "[Desktop Entry]\nIcon=folder-sound\n",
  "/videos": "[Desktop Entry]\nIcon=folder-video\n",
}

class UsersManager extends ConfigFile {
  async login(name) {
    if (!this.value.users.includes(name)) {
      throw new Error(`User ${name} does not exist`)
    }

    const userStyle = `/c/users/${name}/config/style.css`
    if (await fs.access(userStyle)) {
      fileIndex.set("/style.css", userStyle)
    } else {
      fileIndex.set("/style.css", 0)
    }

    await this.update({ activeUser: name })
    location.reload()
  }

  async logout() {
    console.log(this.value.activeUser)
    await this.update({
      activeUser: undefined,
      loggedOut: this.value.activeUser,
    })
    location.reload()
  }

  async createUser(name) {
    if (this.value.users.includes(name)) {
      throw new Error(`User ${name} already exists`)
    }

    // Create the base user folder structure
    const basePath = `/c/users/${name}`

    const folders = Object.keys(DEFAULT_DIRECTORIES)
    for (const folder of folders) {
      await fs.writeDir(`${basePath}${folder}`)
      await fs.writeText(
        `${basePath}${folder}/.directory`,
        DEFAULT_DIRECTORIES[folder],
      )
    }

    await this.update({ users: [...this.value.users, name] })
  }

  async forkUser(name, forkedUserName) {
    if (!this.value.users.includes(name)) {
      throw new Error(`User ${name} does not exist`)
    }
    if (this.value.users.includes(forkedUserName)) {
      throw new Error(`User ${forkedUserName} already exists`)
    }

    const source = `/c/users/${name}`
    const dest = `/c/users/${forkedUserName}`

    await fs.copy(source, dest, { recursive: true })
    await this.update({ users: [...this.value.users, forkedUserName] })
  }

  async chooseUserDialog() {
    document.body.classList.add("desktop")
    await loadDesktopStyle()
    const { prompt, dialog } = await import("../../../ui/layout/dialog.js")
    const { users, loggedOut } = this.value

    let selectedUser = loggedOut ?? users[0]

    const content = {
      tag: ".cols.ma-y-xs.gap-xs",
      content: [
        {
          tag: "select",
          style: {
            minWidth: "23ch",
          },
          tabIndex: 1,
          content: users,
          value: selectedUser,
          on: {
            change: (e) => {
              selectedUser = e.target.value
            },
          },
        },
        {
          tag: "button.shrink",
          tabIndex: 3,
          title: "Create User",
          picto: "user-new",
          on: {
            click: async () => {
              dialogEl.classList.toggle("hide", true)
              const name = await prompt("Enter new user name:", {
                label: "Create User",
              })
              if (name) {
                dialogEl.close()
                await this.createUser(name)
                await this.login(name)
              } else {
                dialogEl.classList.toggle("hide", false)
              }
            },
          },
        },
      ],
    }

    const dialogEl = await dialog({
      label: "Session",
      icon: "apps/session",
      picto: "apps/session",
      skipSave: true,
      maximizable: false,
      closable: false,
      movable: false,
      resizable: false,
      content,
      footer: [
        {
          tag: "button.w-full",
          tabIndex: 2,
          content: "Log In",
          // picto: "key",
          on: { click: () => this.login(selectedUser) },
        },

        // {
        //   tag: "button",
        //   content: "Fork User",
        //   picto: "copy",
        //   on: {
        //     click: async () => {
        //       const forked = await prompt("Which user do you want to fork?")
        //       if (forked) {
        //         const name = await prompt(
        //           `Enter new user name for the fork of ${forked}:`,
        //         )
        //         if (name) {
        //           await this.forkUser(forked, name)
        //           await this.login(name)
        //         }
        //       }
        //     },
        //   },
        // },
      ],
    })
  }

  async claimRootUserDialog() {
    const { prompt, confirm } = await import("../../../ui/layout/dialog.js")

    const accept = await confirm(
      'The default user "windows93" has been updated.\n\nContinuing as "windows93" will overwrite any file in the home folder (including desktop configuration, wallpaper, etc).\n\nAccept changes and overwrite?',
      {
        label: "System Update",
        icon: "warning",
        agree: "Yes, Overwrite",
        decline: "No, keep my files",
      },
    )

    if (accept) {
      const { fileIndex } = await import("../../fileIndex.js")
      await fileIndex.upgrade("/c/users/windows93")
      localStorage.removeItem("sys42_timestamps")
      location.reload()
    } else {
      const name = await prompt("Enter new user name", { label: "Create User" })
      if (name) {
        await this.createUser(name)
        await this.login(name)
      }

      // const fork = await confirm(
      //   'Do you want to fork "windows93" into a new user to keep your files?',
      //   {
      //     label: "System Update",
      //     icon: "question",
      //     agree: "Yes, Fork",
      //     decline: "No, Create new user",
      //   },
      // )

      // const promptLabel = fork
      //   ? "Enter new user name for the fork:"
      //   : "Enter new user name:"
      // const name = await prompt(promptLabel)

      // if (name) {
      //   await (fork ? this.forkUser("windows93", name) : this.createUser(name))
      //   await this.login(name)
      // }
    }
  }
}

export const usersManager = new UsersManager("/users.json5", DEFAULTS)
usersManager.init()
